import pandas as pd
import tkinter as tk
from tkinter import filedialog, messagebox, ttk
import re

class ExcelGrouperApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Excel Grouper Tool")
        self.root.geometry("650x700")

        self.df = None
        self.check_vars_group = []
        self.check_vars_max = []

        tk.Button(root, text="📂 Carica file Excel", command=self.load_excel).pack(pady=10)
        self.file_label = tk.Label(root, text="Nessun file caricato", fg="gray")
        self.file_label.pack()

        self.frame_group = tk.LabelFrame(root, text="Campi su cui raggruppare", padx=10, pady=5)
        self.frame_group.pack(padx=10, pady=10, fill="x")

        self.frame_max = tk.LabelFrame(root, text="Campi su cui prendere il massimo", padx=10, pady=5)
        self.frame_max.pack(padx=10, pady=10, fill="x")

        tk.Label(root, text="Campo su cui sommare:").pack(pady=5)
        self.sum_var = ttk.Combobox(root, width=47, state="readonly")
        self.sum_var.pack()

        tk.Button(root, text="▶ Esegui elaborazione", command=self.process_data,
                  bg="#4CAF50", fg="white").pack(pady=20)

        self.preview_frame = tk.LabelFrame(root, text="Anteprima risultato (prime 10 righe)", padx=10, pady=5)
        self.preview_frame.pack(padx=10, pady=10, fill="both", expand=True)

        self.text_preview = tk.Text(self.preview_frame, height=10, wrap="none")
        self.text_preview.pack(fill="both", expand=True)

    def normalize_columns(self, df):
        """
        Rinomina automaticamente le colonne in forma standardizzata:
        - lowercase
        - sostituisce spazi, trattini, e underscore con '_'
        - rimuove caratteri speciali
        """
        df = df.rename(columns=lambda c: re.sub(r'[^a-z0-9_]', '', c.lower().replace(" ", "_").replace("-", "_")))
        return df

    def load_excel(self):
        file_path = filedialog.askopenfilename(filetypes=[("Excel files", "*.xlsx *.xls")])
        if not file_path:
            return

        try:
            self.df = pd.read_excel(file_path)
            self.df = self.normalize_columns(self.df)

            self.file_label.config(text=f"File caricato: {file_path.split('/')[-1]}", fg="black")

            cols = list(self.df.columns)
            self.sum_var["values"] = cols

            for widget in self.frame_group.winfo_children():
                widget.destroy()
            for widget in self.frame_max.winfo_children():
                widget.destroy()
            self.check_vars_group.clear()
            self.check_vars_max.clear()

            for c in cols:
                var_g = tk.BooleanVar()
                var_m = tk.BooleanVar()
                tk.Checkbutton(self.frame_group, text=c, variable=var_g).pack(anchor="w")
                tk.Checkbutton(self.frame_max, text=c, variable=var_m).pack(anchor="w")
                self.check_vars_group.append((c, var_g))
                self.check_vars_max.append((c, var_m))

        except Exception as e:
            messagebox.showerror("Errore", f"Impossibile leggere il file:\n{e}")

    def _convert_date_like_columns(self, df):
        """Converte automaticamente le colonne con 'date' o 'data' nel nome in datetime"""
        for col in df.columns:
            if "date" in col.lower() or "data" in col.lower():
                try:
                    df[col] = pd.to_datetime(df[col], errors="coerce")
                except Exception:
                    pass
        return df

    def get_column_by_pattern(self, df, pattern):
        """Cerca colonne che corrispondono a un pattern flessibile (es. 'id_treat' trova anche 'id treat', 'ID-TREAT', ecc.)"""
        for col in df.columns:
            if re.search(pattern, col, re.IGNORECASE):
                return col
        return None

    def process_data(self):
        if self.df is None:
            messagebox.showwarning("Attenzione", "Devi prima caricare un file Excel.")
            return

        group_cols = [col for col, var in self.check_vars_group if var.get()]
        max_cols = [col for col, var in self.check_vars_max if var.get()]
        sum_col = self.sum_var.get()

        if not group_cols or not max_cols or not sum_col:
            messagebox.showwarning("Attenzione", "Seleziona tutti i campi richiesti.")
            return

        df_work = self.df.copy()
        df_work = self._convert_date_like_columns(df_work)

        agg_dict = {col: "max" for col in max_cols}
        agg_dict[sum_col] = "sum"

        try:
            df_grouped = df_work.groupby(group_cols, as_index=False).agg(agg_dict)

            # Trova le colonne corrette anche se hanno nomi variabili
            id_treat_col = self.get_column_by_pattern(df_grouped, r"id[_ ]*treat")
            date_supply_col = self.get_column_by_pattern(df_grouped, r"date[_ ]*of[_ ]*supply|data[_ ]*fornitura")

            if id_treat_col:
                df_grouped[id_treat_col] = df_grouped[id_treat_col].astype(str).str.strip()
            if date_supply_col:
                df_grouped[date_supply_col] = pd.to_datetime(df_grouped[date_supply_col], errors="coerce")

            # Ordinamento corretto
            if id_treat_col and date_supply_col:
                df_grouped = df_grouped.sort_values(
                    by=[id_treat_col, date_supply_col],
                    ascending=[True, False],
                    na_position="last"
                ).reset_index(drop=True)
            elif id_treat_col:
                df_grouped = df_grouped.sort_values(id_treat_col, ascending=True).reset_index(drop=True)
            elif date_supply_col:
                df_grouped = df_grouped.sort_values(date_supply_col, ascending=False).reset_index(drop=True)

            # Anteprima
            self.text_preview.delete("1.0", tk.END)
            preview = df_grouped.head(10).copy()
            for c in preview.columns:
                if pd.api.types.is_datetime64_any_dtype(preview[c]):
                    preview[c] = preview[c].dt.strftime("%Y-%m-%d")
            self.text_preview.insert(tk.END, preview.to_string(index=False))

            # Salvataggio
            save_path = filedialog.asksaveasfilename(defaultextension=".xlsx",
                                                     filetypes=[("Excel files", "*.xlsx")],
                                                     title="Salva risultato come")
            if save_path:
                df_grouped.to_excel(save_path, index=False)
                messagebox.showinfo("Successo", f"File salvato correttamente:\n{save_path}")

        except Exception as e:
            messagebox.showerror("Errore", f"Si è verificato un errore:\n{e}")


if __name__ == "__main__":
    root = tk.Tk()
    app = ExcelGrouperApp(root)
    root.mainloop()
